# (c) cavaliba.com - test - test_pipeline.py


from io import StringIO
#import io

import yaml 

from django.core.management import call_command
from django.test import TestCase

import app_home.cache as cache
from app_home.configuration import get_configuration

from app_data.models import DataClass
from app_data.models import DataInstance


from app_data.data import Instance
from app_data.loader import load_broker

from app_data.pipeline import list_pipelines
from app_data.pipeline import get_pipeline
from app_data.pipeline import apply_pipeline



class CommandTest(TestCase):

    fixtures = ["test"]
    def setUp(self):
        cache.init()



    def test_initial_instance_exists(self):

        instance = Instance.load_from_names(classname="test1", keyname="test1-01")
        self.assertEqual(instance.displayname,"test1-01")



    def test_list_pipelines(self):


        # pipelines = list_pipelines()
        # self.assertEqual ( len(pipelines), 5)

        classobj = DataClass.objects.filter(keyname='_pipeline').first()

        DataInstance.objects.filter(classobj=classobj).all().delete()

        pipelines = list_pipelines()
        self.assertEqual ( len(pipelines), 0)


        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              is_enabled: True
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - field_noop                                                   
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        pipelines = list_pipelines()
        self.assertEqual ( len(pipelines), 1)

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest2
              displayname: unittest_pipeline
              description: unittest pipeline
              is_enabled: False
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - field_noop                                                   
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)


        pipelines = list_pipelines(is_enabled=True)
        self.assertEqual ( len(pipelines), 1)

        pipelines = list_pipelines(is_enabled=False)
        self.assertEqual ( len(pipelines), 2)



    def test_create_pipeline(self):

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - field_noop                                                   
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        instance = Instance.load_from_names(classname="_pipeline", keyname="unittest")
        self.assertTrue(instance.is_bound)

        pipeline_data = get_pipeline("unittest")
        self.assertEqual(pipeline_data["tasks"],["field_noop"])

        #print(pipeline_data)
        # {'csv_delimiter': '|', 'classname': 'test1', 'keyfield': 'keyname', 'tasks': ['field_noop']}

        
        
    def test_pipeline_noop(self):

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - field_noop: []
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        datalist = [ {"a":5},{"a":6} ]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertEqual(datalist, [ {"a":5},{"a":6} ])


    def test_pipeline_toint(self):

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - field_toint: ['', a]
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        datalist = [{'a':5}]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertEqual(datalist[0]['a'], 5)

        datalist = [{'a':6.5}]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertEqual(datalist[0]['a'], 6)

        datalist = [{'a':'test'}]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertIsNone(datalist[0]['a'])


    def test_pipeline_tofloat(self):

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - field_tofloat: ['', a]
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        datalist = [{'a':5}]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertEqual(datalist[0]['a'], 5.0)

        datalist = [{'a':6.5}]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertEqual(datalist[0]['a'], 6.5)

        datalist = [{'a':'test'}]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertIsNone(datalist[0]['a'])


    def test_pipeline_tostring(self):

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - field_tostring: ['',a]
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        datalist = [{'a':5}]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertEqual(datalist[0]['a'], '5')

        datalist = [{'a':6.5}]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertEqual(datalist[0]['a'], '6.5')

        datalist = [{'a':'test'}]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertEqual(datalist[0]['a'], 'test')

        datalist = [{'a':True}]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertEqual(datalist[0]['a'], 'True')



    def test_pipeline_set(self):

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - field_set: ['', 'a', '']
                  - field_set: ['', 'b', 5.0]
                  - field_set: ['', 'c', 'test']
                  - field_set: ['', 'd']
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        datalist = [{}]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertEqual(datalist[0]['a'], '')
        self.assertIn('a',datalist[0])
        self.assertEqual(datalist[0]['b'], 5.0)
        self.assertEqual(datalist[0]['c'], 'test')
        self.assertNotIn('d',datalist[0])



    def test_pipeline_copy(self):

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - field_copy: ['', 'a', 'b']
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        datalist = [{'a':5}]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertIn('b',datalist[0])
        self.assertEqual(datalist[0]['b'], 5)



    def test_pipeline_rename(self):

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - field_rename: ['', 'a', 'b']
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        datalist = [{'a':5}]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertNotIn('a',datalist[0])
        self.assertIn('b',datalist[0])
        self.assertEqual(datalist[0]['b'], 5)



    def test_pipeline_delete(self):

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - field_delete: ['',a]
                  - field_delete: ['',b]
                  - field_delete: ['', 'c', 'd']
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        datalist = [{'a':5, 'b':5, 'c':5, 'd':5, 'e':5 } ]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertNotIn('a',datalist[0])
        self.assertNotIn('b',datalist[0])
        self.assertNotIn('c',datalist[0])
        self.assertNotIn('d',datalist[0])
        self.assertIn('e',datalist[0])
        self.assertEqual(datalist[0]['e'], 5)




    def test_pipeline_keep(self):

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - field_keep: ['',a]
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        datalist = [{'a':5, 'b':5, 'c':5, 'd':5 } ]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertNotIn('b',datalist[0])
        self.assertNotIn('c',datalist[0])
        self.assertNotIn('d',datalist[0])
        self.assertEqual(datalist[0]['a'], 5)



    def test_pipeline_upper(self):

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - field_upper: ['',a]
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        datalist = [{'a': 'test'} ]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertIn('a',datalist[0])
        self.assertEqual(datalist[0]['a'], 'TEST')



    def test_pipeline_lower(self):

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - field_lower: ['',a]
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        datalist = [{'a': 'TEST'} ]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertIn('a',datalist[0])
        self.assertEqual(datalist[0]['a'], 'test')


    def test_pipeline_datetime(self):

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - field_date_now: ['',a]
                  - field_datetime_now: ['',b]
                  - field_time_now: ['',c]
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        datalist = datalist = [{'z': 'test'} ]
        apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertIn('a',datalist[0])
        self.assertIn('b',datalist[0])
        self.assertIn('c',datalist[0])
        self.assertTrue(datalist[0]['a'].startswith('20'))
        self.assertTrue(datalist[0]['b'].startswith('20'))
        self.assertEqual( len(datalist[0]['a']), 10)
        self.assertEqual( len(datalist[0]['b']), 19)
        self.assertEqual( len(datalist[0]['c']), 8)
        


    def test_pipeline_regexp_sub(self):

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - field_regexp_sub: [ '', 'a', 'test', 'QWERTY']
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        datalist = [
            {'a': 'This is a test from unittest !'},
            ]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertIn('a',datalist[0])
        self.assertEqual( datalist[0]['a'], "This is a QWERTY from unitQWERTY !")
        


    def test_pipeline_uuid(self):

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - field_uuid: ['',b]
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        datalist = [{'a':5} ]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertIn('a',datalist[0])
        self.assertIn('b',datalist[0])
        self.assertEqual( len(datalist[0]['b']), 36)



    def test_pipeline_merge(self):

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - field_merge: ['','a','b','c']
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        datalist = [
            {'a':5, 'b':6},
            {'a':'unit', 'b':'test'},
        ]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertIn('c',datalist[0])
        self.assertEqual( datalist[0]['c'], 11)
        self.assertIn('c',datalist[1])
        self.assertEqual( datalist[1]['c'], 'unittest')


    def test_pipeline_discard(self):

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - discard: ['']
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        datalist = [
            {'a':5, 'b':6},
            {'a':'unit', 'b':'test'},
        ]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertEqual( len(datalist), 0)



    def test_pipeline_append(self):

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - field_append: ['', a, '-suffix']
                  - field_append: ['', b, '-suffix']
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        datalist = [
            {'a':'test', 'b':6},
        ]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertIn('a',datalist[0])
        self.assertEqual( datalist[0]['a'], 'test-suffix')
        self.assertIn('b',datalist[0])
        self.assertEqual( datalist[0]['b'], '6-suffix')




    def test_pipeline_prepend(self):

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - field_prepend: ['', a, 'prefix-']
                  - field_prepend: ['', b, 'prefix-']
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        datalist = [
            {'a':'test', 'b':6},
        ]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertIn('a',datalist[0])
        self.assertEqual( datalist[0]['a'], 'prefix-test')
        self.assertIn('b',datalist[0])
        self.assertEqual( datalist[0]['b'], 'prefix-6')


 # with conditions

    def test_pipeline_condition(self):

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - set_condition: ['CONDITION_TEST', field_match, a, 'test']
                  - field_set: ['CONDITION_TEST', b, 'testok']
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        datalist = [
            {'a':'test'},
        ]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertIn('a',datalist[0])
        self.assertIn('b',datalist[0])
        self.assertEqual( datalist[0]['b'], 'testok')


    def test_pipeline_condition_negate(self):

        datalist = yaml.safe_load('''
            - classname: _pipeline
              keyname: unittest
              displayname: unittest_pipeline
              description: unittest pipeline
              content: |
                  csv_delimiter: '|'
                  classname: test1
                  keyfield: keyname
                  tasks: 
                  - set_condition: [CONDITION_TEST, field_match, a, 'test']
                  - field_set: ['!CONDITION_TEST', b, 'testok']
            ''')
        aaa = {'perms':['p_pipeline_create']}
        load_broker(datalist=datalist, aaa=aaa, verbose=False)

        datalist = [
            {'a':'unit'},
        ]
        datalist = apply_pipeline(pipeline="unittest", datalist = datalist)
        self.assertIn('a',datalist[0])
        self.assertIn('b',datalist[0])
        self.assertEqual( datalist[0]['b'], 'testok')




# NEXT : load w/ complex pipeline