# (c) cavaliba.com - tests data export

from django.test import override_settings
from django.test import TestCase
from django.test import TransactionTestCase
from django.urls import reverse
from django.core.cache import cache as cache_django
from django.core.management import call_command
from django.conf import settings
import os
import hashlib

from tests import helper


import app_home.cache as cache
from app_user.models import SireneUser
from app_user.role import role_get_by_name


class DataExportTest(TestCase):

    fixtures = ["test"]

    def setUp(self):
        user = SireneUser.objects.create(
            login="unittest", firstname="unittestname")
        user.save()
        role = role_get_by_name("role_admin")
        role.users.add(user)
        role.save()
        cache.clear()

    @override_settings(CAVALIBA_AUTH_MODE="unittest")
    def test_data_export_csv(self):
        response = self.client.post(reverse('app_data:data_export'), {
            'classname': 'test1',
            'page': 'allpages',
            'dv': '___ALL___',
            'query': '',
            'format': 'csv',
        })

        self.assertEqual(response.status_code, 200)
        self.assertContains(response, "test1-01")
        # print(response.content)

    @override_settings(CAVALIBA_AUTH_MODE="unittest")
    def test_data_export_yaml(self):
        response = self.client.post(reverse('app_data:data_export'), {
            'classname': 'test1',
            'page': 'allpages',
            'dv': '___ALL___',
            'query': '',
            'format': 'yaml',
        })

        self.assertEqual(response.status_code, 200)
        self.assertContains(response, "test1-01")
        # print(response.content)

    @override_settings(CAVALIBA_AUTH_MODE="unittest")
    def test_data_export_json(self):
        response = self.client.post(reverse('app_data:data_export'), {
            'classname': 'test1',
            'page': 'allpages',
            'dv': '___ALL___',
            'query': '',
            'format': 'json',
        })

        self.assertEqual(response.status_code, 200)
        self.assertContains(response, "test1-01")
        # print(response.content)

    # with dataview

    @override_settings(CAVALIBA_AUTH_MODE="unittest")
    def test_data_export_yaml_dataview1(self):
        response = self.client.post(reverse('app_data:data_export'), {
            'classname': 'test1',
            'page': 'allpages',
            'dv': 'test1_enumerate_inject',
            'query': '',
            'format': 'yaml',
        })

        self.assertEqual(response.status_code, 200)
        self.assertContains(response, "test1-01")
        self.assertContains(response, "Widget")
        # print(response.content)

    @override_settings(CAVALIBA_AUTH_MODE="unittest")
    def test_data_export_yaml_dataview2(self):
        response = self.client.post(reverse('app_data:data_export'), {
            'classname': 'test1',
            'page': 'allpages',
            'dv': 'test1_welcome',
            'query': '',
            'format': 'yaml',
        })

        # print(response.content)
        self.assertEqual(response.status_code, 200)
        self.assertContains(response, "test1-01")
        self.assertContains(response, "invalid")

    @override_settings(CAVALIBA_AUTH_MODE="unittest")
    def test_data_export_json_dataview(self):
        response = self.client.post(reverse('app_data:data_export'), {
            'classname': 'test1',
            'page': 'allpages',
            'dv': 'test1_welcome',
            'query': '',
            'format': 'json',
        })

        self.assertEqual(response.status_code, 200)
        self.assertContains(response, "test1-01")
        # print(response.content)

    @override_settings(CAVALIBA_AUTH_MODE="unittest")
    def test_export_viewer_get(self):
        """Test GET request to export viewer with admin permission - should show file list"""
        response = self.client.get(reverse('app_data:data_export'))

        self.assertEqual(response.status_code, 200)
        self.assertContains(response, "Export Viewer")

    @override_settings(CAVALIBA_AUTH_MODE="unittest")
    def test_export_viewer_file_list_and_view(self):
        """Test export viewer: create file, list it, download it, and view its content"""
        # Step 1: Create a test file in the export folder
        export_folder = getattr(settings, 'CAVALIBA_EXPORT_FOLDER', '/app/files/export/')
        os.makedirs(export_folder, exist_ok=True)
        test_file = os.path.join(export_folder, 'unittest.txt')
        test_content = 'This is a test export file\n'
        with open(test_file, 'w') as f:
            f.write(test_content)

        try:
            # Step 2: GET the export viewer to list files
            response = self.client.get(reverse('app_data:data_export'))
            self.assertEqual(response.status_code, 200)

            # Step 3: Check for the file entry in the table
            self.assertContains(response, 'unittest.txt')

            # Calculate the expected MD5 hash
            filename_md5 = hashlib.md5('unittest.txt'.encode()).hexdigest()

            # Step 4: Follow the download URL and check for FileResponse
            download_response = self.client.get(
                reverse('app_data:data_export') + f'?dl={filename_md5}'
            )
            self.assertEqual(download_response.status_code, 200)
            # Check that it's a file response with proper headers
            self.assertIn('Content-Disposition', download_response)
            self.assertIn('attachment', download_response['Content-Disposition'])

            # Step 5: Follow the view URL and check for file content
            view_response = self.client.get(
                reverse('app_data:data_export') + f'?view={filename_md5}'
            )
            self.assertEqual(view_response.status_code, 200)
            self.assertContains(view_response, test_content.strip())
            self.assertContains(view_response, 'unittest.txt')  # filename in title

        finally:
            # Cleanup: remove the test file
            if os.path.exists(test_file):
                os.remove(test_file)
