# (c) cavaliba.com - tests / data / view

from django.test import override_settings
from django.test import TestCase
from django.test import TransactionTestCase
from django.urls import reverse
from django.core.cache import cache as cache_django

from tests import helper
import app_home.cache as cache


class DataViews(TestCase):

    fixtures = ["init"]

    def setUp(self):
        helper.add_admin_user(login="unittest")
        cache.clear()
        cache_django.clear()

    # data views
    # ----------

    # Test authenticated access to data private view
    @override_settings(CAVALIBA_AUTH_MODE="unittest")
    def test_data_private(self):
        response = self.client.get(reverse('app_data:private'), follow=True)
        self.assertEqual(response.status_code, 200)


    # Test unauthenticated access redirects to login
    @override_settings(CAVALIBA_AUTH_MODE="local")
    def test_data_private_unauthenticated(self):
        self.client.logout()
        response = self.client.get(reverse('app_data:private'))
        self.assertEqual(response.status_code, 302)


    # Test listing site instances
    @override_settings(CAVALIBA_AUTH_MODE="unittest")
    def test_instance_list_site(self):
        helper.add_test_sites(count=5)

        response = self.client.get(
            reverse('app_data:instance_list', kwargs={'classname': 'site'}),
            follow=True)
        self.assertEqual(response.status_code, 200)

        self.assertContains(response, 'site01')
        self.assertContains(response, 'site02')
        self.assertContains(response, 'site03')
        self.assertContains(response, 'site04')
        self.assertContains(response, 'site05')


    # Test editing a site instance
    @override_settings(CAVALIBA_AUTH_MODE="unittest")
    def test_instance_edit_site(self):
        helper.add_test_sites(count=1)

        from app_data.models import DataInstance
        site = DataInstance.objects.filter(classname='site', keyname='site01').first()
        self.assertIsNotNone(site)

        response = self.client.get(
            reverse('app_data:instance_edit', kwargs={'id': site.id}),
            follow=True)
        self.assertEqual(response.status_code, 200)
        self.assertContains(response, 'csrfmiddlewaretoken')

        post_data = {
            'keyname': 'site01',
            'displayname': 'Updated Site 01',
            'description': 'Test site description',
        }
        response = self.client.post(
            reverse('app_data:instance_edit', kwargs={'id': site.id}),
            data=post_data,
            follow=True)
        self.assertEqual(response.status_code, 200)

        site_updated = DataInstance.objects.get(id=site.id)
        self.assertEqual(site_updated.displayname, 'Updated Site 01')


    # Test creating a new site instance
    @override_settings(CAVALIBA_AUTH_MODE="unittest")
    def test_instance_new_site(self):
        response = self.client.get(
            reverse('app_data:instance_new', kwargs={'classname': 'site'}),
            follow=True)
        self.assertEqual(response.status_code, 200)
        self.assertContains(response, 'csrfmiddlewaretoken')

        post_data = {
            'keyname': 'newsite99',
            'displayname': 'New Site 99',
            'description': 'A brand new test site',
        }
        response = self.client.post(
            reverse('app_data:instance_new', kwargs={'classname': 'site'}),
            data=post_data,
            follow=True)
        self.assertEqual(response.status_code, 200)

        from app_data.models import DataInstance
        new_site = DataInstance.objects.filter(classname='site', keyname='newsite99').first()
        self.assertIsNotNone(new_site)
        self.assertEqual(new_site.displayname, 'New Site 99')


    # Test viewing site instance details
    @override_settings(CAVALIBA_AUTH_MODE="unittest")
    def test_instance_detail_site(self):
        helper.add_test_sites(count=1)

        from app_data.models import DataInstance
        site = DataInstance.objects.filter(classname='site', keyname='site01').first()
        self.assertIsNotNone(site)

        response = self.client.get(
            reverse('app_data:instance_detail', kwargs={'id': site.id}),
            follow=True)
        self.assertEqual(response.status_code, 200)

        self.assertContains(response, 'site01')
        self.assertContains(response, 'Site 01')


    # Test deleting a site instance
    @override_settings(CAVALIBA_AUTH_MODE="unittest")
    def test_instance_delete_site(self):
        helper.add_test_sites(count=1)

        from app_data.models import DataInstance
        site = DataInstance.objects.filter(classname='site', keyname='site01').first()
        self.assertIsNotNone(site)
        site_id = site.id

        response = self.client.post(
            reverse('app_data:instance_delete', kwargs={'id': site_id}),
            follow=True)
        self.assertEqual(response.status_code, 200)

        deleted_site = DataInstance.objects.filter(id=site_id).first()
        self.assertIsNone(deleted_site)