# (c) cavaliba.com - IAM - permission.py

import re
import json
import yaml

from django.http import HttpResponse
from django.forms.models import model_to_dict
from django.utils.translation import gettext as _
from django.utils import timezone

from app_home.log import log, DEBUG, INFO, WARNING, ERROR, CRITICAL
from .models import SirenePermission

from app_data.permissions import has_permission_read_permission
from app_data.permissions import has_permission_create_permission
from app_data.permissions import has_permission_update_permission
from app_data.permissions import has_permission_delete_permission



def permission_get_all():
    return SirenePermission.objects.all()



def permission_get_by_id(pid):
    return SirenePermission.objects.filter(pk=pid).first()


def permission_get_by_name(keyname):
    return SirenePermission.objects.filter(keyname=keyname).first()


def permission_get_by_data(data):
    
    keyname = data.get('keyname', None)
    if not keyname:
        return
    return  permission_get_by_name(keyname)




# def permission_set(keyname=None, displayname=None, description=None):
    
#     ''' create/update a single permission in Database '''

#     if not keyname:
#         return

#     p = permission_get_by_name(keyname)
#     if not p:
#         p = SirenePermission()
#         p.keyname = keyname

#     if displayname:
#         p.displayname = displayname

#     if description:        
#         p.description = description

#     p.save()

#     return p




def permission_init(data):

    keyname = data.get('keyname', None)
    if not keyname:
        return

    # skip if exists
    entry = permission_get_by_name(keyname)
    if entry:
        return entry

    entry = SirenePermission()
    entry = permission_update(entry, data)

    return entry



def permission_create(data):

    keyname = data.get('keyname', None)
    if not keyname:
        return

    # create or update
    entry = permission_get_by_name(keyname)
    if not entry:
        entry = SirenePermission()

    entry = permission_update(entry, data)

    return entry


def permission_update(iobj, data):
    ''' update Object with data dict info'''

    attributs =  [
        "keyname", "displayname", "description",
    ] 

    if not iobj:
        return

    if not data:
        return

    for attrib in attributs:
        if attrib in data:
            try:
                setattr(iobj, attrib, data[attrib])
            except:
                pass

    iobj.last_update = timezone.now()

    iobj.save()
    return iobj


def permission_update_by_data(data):

    iobj = permission_get_by_data(data)
    if iobj:
        return permission_update(iobj, data)



def permission_delete(iobj):

    if not iobj:
        return False

    if iobj.is_builtin:
        return False
    
    try:
        iobj.delete()
        return True
    except Exception as e:
        print("permission_delete failed : ",e)
        return False


def permission_delete_by_id(pid):

    iobj = permission_get_by_id(pid)
    permission_delete(iobj)
    return iobj



def permission_delete_by_data(data):

    iobj = permission_get_by_data(data)
    if iobj:
        return permission_delete_by_id(iobj.id)


# -------------------------------------------------------
# IMPORTS
# -------------------------------------------------------

# when bootstraping Cavaliba, no permission exists yet
# create first permission before (home/)
def bootstrap_permissions(datadict):

    # data = {
    #     "keyname": "p_permission_create",
    #     "description": "Built-in IAM permission : create (bootstrap)",
    #     "displayname": "p_permission_create"
    #     [_action: delete]
    # }


    for data in datadict:

        keyname = data.get("keyname", None)
        if not keyname: 
            continue


        if '_action' in data:
            if data['_action'] == 'delete':
                try:
                    SirenePermission.objects.filter(keyname=keyname).delete()
                except Exception as e:
                    print("ERR: ",e)
                continue


        entry = SirenePermission.objects.filter(keyname=keyname).first()

        if not entry:
            entry = SirenePermission()
        
        entry.is_builtin = True
        try:
            entry.keyname = data['keyname']
            entry.displayname = data['keyname']
        except:
            pass
        try:
            entry.description = data['description']
        except:
            pass

        entry.save()

    count = len(datadict)
    log(WARNING, app="iam", view="permission", action="bootstrap", status="OK", data=f"{count} permissions") 


    return count



def load_permission(datadict=None, verbose=None, aaa=None):

    if not datadict:
        return

    if not aaa:
        log(WARNING, aaa=aaa, app="iam", view="permission", action="load", status="DENY", data=_("Not allowed")) 
        return


    keyname = datadict.get("keyname", None)
    if not keyname: 
        return

    action = datadict.get("_action", "create")

    r = None

    # if action == "init_builtin":
    #     r = permission_init(datadict)
    if action == "init":
        if not has_permission_create_permission(aaa=aaa):
            log(WARNING, aaa=aaa, app="iam", view="permission", action="init", status="DENY", data=_("Not allowed")) 
            return        
        r = permission_init(datadict)

    elif action == "create":
        if not has_permission_create_permission(aaa=aaa):
            log(WARNING, aaa=aaa, app="iam", view="permission", action="create", status="DENY", data=_("Not allowed")) 
            return        
        r = permission_create(datadict)

    elif action == "update":
        if not has_permission_update_permission(aaa=aaa):
            log(WARNING, aaa=aaa, app="iam", view="permission", action="update", status="DENY", data=_("Not allowed")) 
            return        
        r = permission_update_by_data(datadict)

    elif action == "delete":
        if not has_permission_delete_permission(aaa=aaa):
            log(WARNING, aaa=aaa, app="iam", view="permission", action="delete", status="DENY", data=_("Not allowed"))
            return        
        r = permission_delete_by_data(datadict)

    else:
        pass

    if r:
        log(INFO, aaa=aaa, app="iam", view="permission", action=action, status="OK", data=keyname)
    else:
        log(INFO, aaa=aaa, app="iam", view="permission", action=action, status="KO", data=keyname)
    if verbose:
        print(f"permission: {action} - {keyname} - {r}")

        
    return r

# -------------------------------------------------------
# EXPORT
# -------------------------------------------------------

def permission_listdict_format(permissions):
    ''' list of  Models to  list of dict '''

    dict_attributs =  ["keyname", "is_builtin", "displayname", "is_enabled","description" ] 

    datalist = []
    for permission in permissions:
        m = model_to_dict(permission, fields=dict_attributs)
        m["classname"] = "_permission"

        # remove null values
        m2= {}
        for k,v in m.items():
            if not isinstance(v,bool):
                if v:
                    m2[k] = v
            else:
                m2[k] = v

        datalist.append(m2)

    return datalist


# JSON
def permission_json_response(items):
    datalist = permission_listdict_format(items)
    filedata = json.dumps(datalist, indent=4, ensure_ascii = False)
    response = HttpResponse(filedata, content_type='text/json')  
    response['Content-Disposition'] = 'attachment; filename="permissions.json"'
    return response

# YAML
class MyYamlDumper(yaml.SafeDumper):
    def write_line_break(self, data=None):
        super().write_line_break(data)
        if len(self.indents) < 2:
            super().write_line_break()


def permission_yaml_response(items):
    datalist = permission_listdict_format(items)
    filedata = yaml.dump(datalist, allow_unicode=True, Dumper=MyYamlDumper, sort_keys=False)
    response = HttpResponse(filedata, content_type='text/yaml')  
    response['Content-Disposition'] = 'attachment; filename="permissions.yaml"'
    return response
