#!/bin/bash

# (c) cavaliba.com - exporter.sh

# SETUP:  choose run mode (auto cron / manual)
#  --------------------------------------------
# cron mode (no TTY in docker command)
# 32 15 * * * /opt/cavaliba/script/exporter.sh /opt/cavaliba > /var/log/cavaliba_exporter.log 2>&1
#CAVALIBA="docker exec -i cavaliba_app python manage.py cavaliba "

# manual mode (TTY in docker command) ; from cavaliba home dir
CAVALIBA="docker exec -it cavaliba_app python manage.py cavaliba "

#TIMESTAMP=$(date +"%Y-%m-%d-%H-%M-%S")
TIMESTAMP=$(date +"%Y-%m-%d")

# relative to HOMEDIR
HOMEDIR="${1:-.}"
EXPORT_FOLDER="files/export/"
LOGFILE="$EXPORT_FOLDER/000_log.log"
KEEPDAYS=2

# Function to log messages with timestamp
log() {
	local level="$1"
	local message="$2"
	local current_time=$(date +"%Y-%m-%d %H:%M:%S")
	local log_output="[$current_time] [$level] $message"
	echo "$log_output"
	echo "$log_output" >> $LOGFILE
}


# Function to move old files to _old folder
cleanup_old_files() {
	local old_folder="$EXPORT_FOLDER/_old"

	if [ ! -d "$old_folder" ]; then
		log "INFO" "Creating _old folder: $old_folder"
		mkdir -p "$old_folder"
	fi

	log "INFO" "Moving files older than $KEEPDAYS days to $old_folder"

	find "$EXPORT_FOLDER" -maxdepth 1 -type f -mtime +$KEEPDAYS | while read -r file; do
		filename=$(basename "$file")
		# Ignore files starting with a dot
		if [[ "$filename" == .* ]]; then
			continue
		fi
		log "INFO" "Moving old file: $filename"
		mv "$file" "$old_folder/"
	done
}

# Function to export a schema
# Parameters: schema_name
export_schema() {
	local schema_name="$1"

	if [ -z "$schema_name" ]; then
		log "ERROR" "schema name required"
		return 1
	fi

	log "INFO" "Exporting schema: $schema_name to JSON"
	$CAVALIBA get --schema "$schema_name" --last 999999999 --json > $EXPORT_FOLDER/"$schema_name"-latest.json
	cp $EXPORT_FOLDER/"$schema_name"-latest.json $EXPORT_FOLDER/"$schema_name"-$TIMESTAMP.json

	log "INFO" "Exporting schema: $schema_name to YAML"
	$CAVALIBA get --schema "$schema_name" --last 999999999 --yaml > $EXPORT_FOLDER/"$schema_name"-latest.yaml
	cp $EXPORT_FOLDER/"$schema_name"-latest.json $EXPORT_FOLDER/"$schema_name"-$TIMESTAMP.yaml

}

#  ----------------------------------------------------------------
# entry point
#  ----------------------------------------------------------------

# Change to HOMEDIR
cd "$HOMEDIR" || exit 1

# Check for required files and directories
if [ ! -f "$HOMEDIR/.env" ]; then
	echo "Error: .env file not found in $HOMEDIR"
	exit 1
fi

if [ ! -f "$HOMEDIR/docker-compose.yml" ]; then
	echo "Error: docker-compose.yml file not found in $HOMEDIR"
	exit 1
fi

if [ ! -d "$HOMEDIR/files" ]; then
	echo "Error: files/ folder not found in $HOMEDIR"
	exit 1
fi

# Purge logfile at startup if it exists
if [ -f "$LOGFILE" ]; then
	rm -f $LOGFILE
fi

# Log script startup
log "INFO" "Export startup at $(date +"%Y-%m-%d %H:%M:%S")"

# Export schemas
export_schema "app"
export_schema "site"
export_schema "sitegroup"
export_schema "_group"
export_schema "_dataview"
export_schema "_enumerate"
export_schema "_pipeline"
export_schema "_user"
export_schema "unknown"

# Cleanup old files
cleanup_old_files

# Log script completion
log "INFO" "Script done at $(date +"%Y-%m-%d %H:%M:%S")"
