package cmd

import (
	"context"
	"crypto/tls"
	"errors"
	"fmt"
	"io"
	"net/http"
	"time"

	"github.com/spf13/viper"
)

type APITarget struct {
	url string
	//verb       string
	//timeout time.Duration
	ssl_skipverify bool
	body           string
}

type APIResult struct {
	http_code    int
	duration     time.Duration
	content_type string
	body         string
}

// var http_client = &http.Client{
// 	Timeout: 5000 * time.Millisecond,
// }

// Call API
func CallAPI(target APITarget) (APIResult, error) {

	var result APIResult

	transport := &http.Transport{
		TLSClientConfig: &tls.Config{InsecureSkipVerify: target.ssl_skipverify},
	}

	http_client := &http.Client{
		Transport: transport,
		Timeout:   5000 * time.Millisecond,
	}

	//timeout := target.timeout
	timeout := time.Duration(500) * time.Millisecond

	ctx, cancel := context.WithTimeout(context.Background(),
		timeout)
	defer cancel()

	req, err := http.NewRequestWithContext(ctx,
		http.MethodGet,
		target.url,
		nil)
	if err != nil {
		return result, errors.New("invalid server parameters")
	}

	var header = viper.GetString("client_id") + " " + viper.GetString("client_secret")
	req.Header.Add("X-Cavaliba-Key", header)

	start := time.Now()

	resp, err := http_client.Do(req)
	if err != nil {
		return result, errors.New("no answer from server")
	}
	defer resp.Body.Close()

	result.duration = time.Since(start)
	result.http_code = resp.StatusCode
	result.content_type = resp.Header.Get("Content-Type")

	if resp.StatusCode == 401 {
		return result, fmt.Errorf("invalid credentials (401)")
	}
	if resp.StatusCode == 404 {
		return result, fmt.Errorf("not found (404)")
	}
	if resp.StatusCode != 200 {
		return result, fmt.Errorf("http error from server (%d)", resp.StatusCode)
	}

	bodyBytes, err := io.ReadAll(resp.Body)
	if err != nil {
		//log.Println(err)
		return result, errors.New("can't read server answer")
	}

	result.body = string(bodyBytes)

	return result, nil
}

// PrintVerboseTarget displays API call details when verbose flag is enabled
func PrintVerboseTarget(target APITarget) {
	if verbose {
		fmt.Println("----- API Request -----")
		fmt.Println("API:       ", target.url)
		fmt.Println("")
	}
}

// PrintVerboseResult displays API call details when verbose flag is enabled
func PrintVerboseResult(result APIResult) {
	if verbose {
		fmt.Println("----- API Response -----")
		fmt.Println("status:    ", result.http_code)
		fmt.Println("duration:  ", result.duration)
		fmt.Println("")
		fmt.Println("----- DATA -------------")
	}
}

// AppendGlobalOptions appends query parameters to the URL based on global flags
func AppendGlobalOptions(target *APITarget) error {
	separator := "?"

	if expand {
		target.url = target.url + separator + "expand=true"
		separator = "&"
	}

	if search != "" {
		target.url = target.url + separator + "search=" + search
		separator = "&"
	}

	if schema != "" {
		target.url = target.url + separator + "schema=" + schema
		separator = "&"
	}

	if page > 0 {
		target.url = target.url + separator + fmt.Sprintf("page=%d", page)
		separator = "&"
	}

	if size > 0 {
		target.url = target.url + separator + fmt.Sprintf("size=%d", size)
		separator = "&"
	}

	if format == "json" {
		target.url = target.url + separator + "o=json"
		separator = "&"
	} else if format == "yaml" {
		target.url = target.url + separator + "o=yaml"
		separator = "&"
	} else if format == "txt" {
		target.url = target.url + separator + "o=txt"
		separator = "&"
	} else {
		return fmt.Errorf("invalid format: %s (valid formats: json, yaml, txt)", format)
	}

	return nil
}
