/*
Copyright © 2025 cavaliba.com
*/
package cmd

import (
	"fmt"
	"net/http"
	"os"
	"path/filepath"
	"strings"

	"github.com/spf13/cobra"
	"github.com/spf13/viper"
)

// importCmd represents the import command
var importCmd = &cobra.Command{
	Use:   "import",
	Short: "Import data into a Cavaliba instance",
	Long:  `Import schemas and instances from YAML, JSON, or CSV files into a Cavaliba instance`,
	Run: func(cmd *cobra.Command, args []string) {

		if file == "" {
			fmt.Println("Error: --file flag is required for import command")
			os.Exit(1)
		}

		// Check if file exists
		if _, err := os.Stat(file); os.IsNotExist(err) {
			fmt.Printf("Error: file not found: %s\n", file)
			os.Exit(1)
		}

		// Read file contents
		fileContent, err := os.ReadFile(file)
		if err != nil {
			fmt.Printf("Error: failed to read file: %s\n", err)
			os.Exit(1)
		}

		// Detect content type based on file extension
		ext := strings.ToLower(filepath.Ext(file))
		var contentType string
		switch ext {
		case ".yaml", ".yml":
			contentType = "application/x-yaml"
		case ".json":
			contentType = "application/json"
		case ".csv":
			contentType = "text/csv"
		default:
			fmt.Printf("Error: unsupported file type: %s (supported: .yaml, .yml, .json, .csv)\n", ext)
			os.Exit(1)
		}

		endpoint := "import/"

		target := APITarget{
			url:            viper.GetString("url") + endpoint,
			method:         http.MethodPost,
			ssl_skipverify: viper.GetBool("ssl_skipverify"),
			content_type:   contentType,
			body:           string(fileContent),
		}

		err = AppendGlobalOptions(&target)
		if err != nil {
			fmt.Println(err)
			os.Exit(1)
		}

		PrintVerboseTarget(target)

		result, err := CallAPI(target)
		if err != nil {
			PrintError(result, err)
			os.Exit(0)
		}

		PrintVerboseResult(result)
		fmt.Println(result.body)

	},
}

func init() {
	rootCmd.AddCommand(importCmd)
}
