# (c) cavaliba.com - data - views_import.py


import re
import yaml 
import json 
import csv

from django.shortcuts import render, redirect
from django.http import HttpResponse
from django.contrib import messages
from django.utils.translation import gettext as _
from django.db.models import Q
from django.http import FileResponse

from app_home.log import log, DEBUG, INFO, WARNING, ERROR, CRITICAL
from app_user.aaa import start_view
from app_home.configuration import get_configuration
import app_home.cache as cache

from app_data.schema import Schema
from app_data.data import Instance
from app_data.forms import DataUploadForm

from app_data.loader import load_broker
from app_data.loader import load_file_csv
from app_data.loader import load_file_json
from app_data.loader import load_file_yaml

from app_data.pipeline import list_pipelines
from app_data.pipeline import apply_pipeline

from app_data.filestore import tmp_local_filepath

from app_user.models import SireneUser
from app_user.models import SireneGroup
from app_user.user import user_listdict_format

from app_user.group import group_to_yaml
from app_user.role import role_to_yaml

# -------------------------------------------------------------------------
# YAML TEXT EDITOR TOOL (global)
# -------------------------------------------------------------------------
#def texteditor(request, classname=None, keyname=None, id=None):
def texteditor(request, classname=None, id=None):

    context = start_view(request, app="data", view="texteditor", 
        noauth="app_sirene:private", perm="p_data_import", noauthz="app_home:private")
    if context["redirect"]:
        return redirect(context["redirect"])
    aaa = context["aaa"]

    rawdata = ""

    # Get existing data or empty
    if request.method == "GET":

        #  no id          : => schema
        # no classname   : => instance
        # id + classname : _user, _group or _role


        # Schema
        if not id:
            schema = Schema.from_name(classname)
            try:
                rawdata = schema.to_yaml()
            except:
                pass

        # Instance
        elif not classname:
            instance = Instance.from_id(id)
            #instance.print()
            if instance:
                rawdata = instance.to_yaml()  

        #  special case (non Schema struct)
        elif classname == '_user':
            user = SireneUser.objects.filter(id=id).first()
            if user:
                rawdata = user.to_yaml()

        elif classname == '_group':
            group = SireneGroup.objects\
                .filter(is_role=False, id=id)\
                .prefetch_related("users")\
                .prefetch_related("subgroups")\
                .first()
            if group:
                rawdata = group_to_yaml(group)

        elif classname == '_role':
            role = SireneGroup.objects\
                .filter(is_role=True, id=id)\
                .prefetch_related("users")\
                .prefetch_related("subgroups")\
                .prefetch_related("permissions")\
                .first()            
            if role:
                rawdata = role_to_yaml(role)


    elif request.method == "POST":

        valid = False
        rawdata = request.POST["rawdata"]
        datalist = None

        # Get Pipeline
        #  TODO: sanitize
        pipeline = request.POST["pipeline"]


        # Get TextArea
        try:
            datalist = yaml.safe_load(rawdata)
            valid = True
        except Exception as e:
            messages.add_message(request, messages.ERROR, e)

        # File provided ?
        fileform = DataUploadForm(request.POST, request.FILES)
        
        if fileform.is_valid():

            postfile = request.FILES["file"]

            # filestore tmp file
            filename = tmp_local_filepath()
            with open(filename, "wb+") as destination:
                for chunk in postfile.chunks():
                    destination.write(chunk)


            # if file.multiple_chunks():
            #     messages.add_message(request, messages.ERROR, _("Import failed - file too big"))
            #     log(ERROR, aaa=aaa, app="data", view="import", action="post", status="KO", data=_("Import failed - file too big"))
            #     return redirect("app_data:private")

            if postfile.name.endswith('.csv'):
                datalist = load_file_csv(filename=filename, pipeline=pipeline)

            elif postfile.name.endswith('.yml') or postfile.name.endswith('.yaml'):
                datalist = load_file_yaml(filename=filename, pipeline=pipeline)

            elif postfile.name.endswith('.json'):
                datalist = load_file_json(filename=filename, pipeline=pipeline)

            if type(datalist) is not list:
                messages.add_message(request, messages.ERROR, _("Import failed - invalid file"))
                log(ERROR, aaa=aaa, app="data", view="import", action="file", status="KO", 
                    data=f"Content is not a list : {postfile.name}")
                datalist = []
            else:
                log(INFO, aaa=aaa, app="data", view="import", action="file", status="OK", 
                    data=f"loaded: {postfile.name}")
                valid = True


        #  apply pipeline if any
        if valid:        
            datalist = apply_pipeline(pipeline=pipeline, datalist=datalist)


        # check only
        if request.POST["submit"] == "verify":
            if valid:
                messages.add_message(request, messages.SUCCESS, _("Check ok"))
                log(DEBUG, aaa=aaa, app="data", view="import", action="check", status="OK")
                # no redirect, keep editing
            else:
                # error already displayed
                pass

        # submit
        if request.POST["submit"] == "import":
            # load twice for references
            count = load_broker(datalist, aaa=aaa)
            count = load_broker(datalist, aaa=aaa)
            messages.add_message(request, messages.SUCCESS, _("Import OK"))
            log(DEBUG, aaa=aaa, app="data", view="import", action="import", status="OK",
                data=f"imported: {count} objects")
            return redirect("app_home:private")

    # import may often alter permissions, group, roles
    # it's best to flush for this user after such action, to avoid stale info or logout/login action.
    # request.session.flush()

    
    context["upload_form"] = DataUploadForm()
    context["rawdata"] = rawdata
    context["pipelines"] = list_pipelines(is_enabled=True)
    return render(request, 'app_data/import.html', context)

